/* * Copyright (c) 2026 M.Graell
 * * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#include <WiFi.h>
#include <WebServer.h>
#include <Wire.h>
#include <Adafruit_Sensor.h>
#include <Adafruit_BME280.h>
#include <max6675.h>
#include <OneWire.h>
#include <DallasTemperature.h>
#include <ArduinoJson.h>
#include <EEPROM.h>

// Pin Definitionen für ESP32
#define RELAY_1 32
#define RELAY_2 33
#define RELAY_3 25
#define RELAY_4 26
const int relayPins[] = {RELAY_1, RELAY_2, RELAY_3, RELAY_4};

#define MAX6675_SO 19
#define MAX6675_CS 23
#define MAX6675_SCK 18
#define ONE_WIRE_BUS 4
#define I2C_SDA 21
#define I2C_SCL 22

WebServer server(80);
MAX6675 thermocouple(MAX6675_SCK, MAX6675_CS, MAX6675_SO);
Adafruit_BME280 bme;
OneWire oneWire(ONE_WIRE_BUS);
DallasTemperature ds18(&oneWire);

struct Config { char ssid[32]; char pass[64]; int sensorType; int relayCount; } config;
struct Relais { char name[16]; int mode; float ein; float aus; bool aktuellAn; } rList[4];

bool systemAktiv = false;
unsigned long startZeit = 0;
long timerDauerMinuten = 0; 
unsigned long timerEndeMillis = 0;
bool restartRequested = false;
unsigned long restartTimestamp = 0;

const char INDEX_HTML[] PROGMEM = R"=====(<!DOCTYPE html><html lang="de"><head><meta charset="UTF-8"><meta name="viewport" content="width=device-width,initial-scale=1.0"><title>ESP32 BBQ</title><style>:root{--bg:#121212;--card:#1e1e1e;--primary:#ff4500;--text:#e0e0e0}body{font-family:sans-serif;background:var(--bg);color:var(--text);padding:10px;max-width:850px;margin:auto}.card{background:var(--card);padding:15px;border-radius:12px;margin-bottom:15px;border:1px solid #333}table{width:100%;border-collapse:collapse;margin-top:10px}th,td{padding:8px;border-bottom:1px solid #333;text-align:center}button{background:var(--primary);color:#fff;border:none;padding:12px;border-radius:5px;cursor:pointer;width:100%;font-weight:700}input,select{background:#333;color:#fff;border:1px solid #555;padding:6px;border-radius:4px;width:100%;box-sizing:border-box}label{font-size:.85em;color:#888;display:block;margin-top:10px}.grid{display:grid;grid-template-columns:1fr 1fr;gap:10px}#tempSVG{background:#000;border-radius:5px;width:100%;height:150px;margin-top:10px}</style></head><body><div class="card"><h2>Status: <span id="statustxt">LÄDT...</span></h2><div class="grid"><div style="text-align:center;font-size:1.5em;"><span id="curT">0</span>°C</div><div style="text-align:center">Laufzeit: <span id="runtime">00:00:00</span></div></div><div id="timerDisplay" style="text-align:center;margin-top:5px;color:#ff4500"></div><svg id="tempSVG" viewBox="0 0 500 150"><polyline id="poly" fill="none" stroke="#ff4500" stroke-width="2" points=""/></svg><div class="grid" style="margin-top:15px"><button onclick="cmd('/stop')" style="background:#d32f2f">STOP</button><button onclick="cmd('/start')" style="background:#388e3c">START / RESET</button></div></div><div class="card"><h3>Timer (Minuten)</h3><div class="grid"><input type="number" id="tMin" value="0"><button onclick="setTimer()">Timer setzen</button></div></div><div class="card"><h3>Relais & Schwellwerte</h3><table><thead><tr><th>Name</th><th>Modus</th><th>An</th><th>Aus</th><th>Status</th></tr></thead><tbody id="rtable"></tbody></table><button onclick="saveRel()" style="background:#444;margin-top:15px">Werte Speichern</button></div><div class="card"><h3>System</h3><div class="grid"><div><label>Relais:</label><select id="rC"><option value="1">1</option><option value="2">2</option><option value="3">3</option><option value="4">4</option></select></div><div><label>Sensor:</label><select id="sT"><option value="0">BME280</option><option value="1">MAX6675</option><option value="2">DS18B20</option></select></div></div><label>WLAN SSID:</label><input type="text" id="ssid"><label>Passwort:</label><input type="password" id="pass"><button onclick="saveSys()" style="margin-top:15px;background:#222">WLAN & Neustart</button></div><div class="card"><a href="https://www.graell.de" target="_blank" style="color:#888;text-decoration:none">ESP32 BBQ BME280(0x77)</a></div><script>let tempHistory=[],maxPoints=50,firstLoad=!0;async function update(){try{const e=await fetch("/data");if(!e.ok)return;const t=await e.json();document.getElementById("curT").innerText=t.t.toFixed(1),document.getElementById("runtime").innerText=t.run,document.getElementById("statustxt").innerText=t.active?"AKTIV":"GESTOPPT",document.getElementById("statustxt").style.color=t.active?"#0f0":"#f00",document.getElementById("timerDisplay").innerText=t.tLeft?"Restzeit: "+t.tLeft:"",drawChart(t.t);if(firstLoad&&t.rel&&t.rel.length>0){let e="";t.rel.forEach((t,a)=>{e+=`<tr><td><input id="n${a}" value="${t.n}"></td><td><select id="m${a}"><option value="0" ${0==t.m?"selected":""}>Heiz</option><option value="1" ${1==t.m?"selected":""}>Kühl</option></select></td><td><input type="number" step="0.1" id="e${a}" value="${t.e}"></td><td><input type="number" step="0.1" id="o${a}" value="${t.o}"></td><td id="s${a}" style="font-weight:bold">-</td></tr>`}),document.getElementById("rtable").innerHTML=e,document.getElementById("sT")&&(document.getElementById("sT").value=t.conf.sT),document.getElementById("rC")&&(document.getElementById("rC").value=t.conf.rC),document.getElementById("ssid")&&(document.getElementById("ssid").value=t.conf.ssid),firstLoad=!1}firstLoad||t.rel.forEach((e,t)=>{const a=document.getElementById("s"+t);a&&(a.innerText=e.s?"AN":"AUS",a.style.color=e.s?"#0f0":"#f00")})}catch(e){console.error("Update Fehler:",e),document.getElementById("statustxt").innerText="VERBINDUNGSFEHLER"}}function drawChart(e){tempHistory.push(e),tempHistory.length>maxPoints&&tempHistory.shift();let t="",a=Math.min(...tempHistory)-2,n=Math.max(...tempHistory)+2,s=n-a;for(let r=0;r<tempHistory.length;r++){let i=r/(maxPoints-1)*500,l=150-(tempHistory[r]-a)/s*150;t+=i+","+l+" "}document.getElementById("poly").setAttribute("points",t)}function cmd(e){fetch(e),e.includes("start")&&(tempHistory=[])}function setTimer(){fetch("/setTimer?m="+document.getElementById("tMin").value)}function saveRel(){let e="num="+document.getElementById("rtable").rows.length;for(let t=0;t<document.getElementById("rtable").rows.length;t++)e+=`&n${t}=${encodeURIComponent(document.getElementById("n"+t).value)}&m${t}=${document.getElementById("m"+t).value}&e${t}=${document.getElementById("e"+t).value}&o${t}=${document.getElementById("o"+t).value}`;fetch("/saveRel?"+e).then(()=>alert("Gespeichert"))}function saveSys(){const e=`st=${document.getElementById("sT").value}&rc=${document.getElementById("rC").value}&s=${encodeURIComponent(document.getElementById("ssid").value)}&p=${encodeURIComponent(document.getElementById("pass").value)}`;fetch("/saveSys?"+e).then(()=>alert("Neustart..."))}setInterval(update,2000),update();</script></body></html>)=====";

void allOff() { 
  for(int i=0; i<4; i++) { 
    digitalWrite(relayPins[i], LOW); 
    rList[i].aktuellAn = false; 
  } 
}

String formatTime(unsigned long ms) {
  unsigned long s = ms / 1000;
  int h = s / 3600; int m = (s % 3600) / 60; int sec = s % 60;
  char buf[12]; sprintf(buf, "%02d:%02d:%02d", h, m, sec);
  return String(buf);
}

float getTemp() {
  float t = 0;
  if (config.sensorType == 0) t = bme.readTemperature();
  else if (config.sensorType == 1) t = thermocouple.readCelsius();
  else if (config.sensorType == 2) { ds18.requestTemperatures(); t = ds18.getTempCByIndex(0); }
  return (isnan(t) || t < -50) ? 0 : t;
}

void saveToEEPROM() {
  EEPROM.put(0, config);
  for(int i=0; i<4; i++) EEPROM.put(150 + (i*sizeof(Relais)), rList[i]);
  EEPROM.commit();
}

void loadFromEEPROM() {
  EEPROM.get(0, config);
  if (config.sensorType < 0 || config.sensorType > 2) {
    config.sensorType = 0; config.relayCount = 1;
    for(int i=0; i<4; i++) { sprintf(rList[i].name, "R %d", i+1); rList[i].mode = 0; rList[i].ein = 16.0; rList[i].aus = 20.0; }
  } else {
    for(int i=0; i<4; i++) EEPROM.get(150 + (i*sizeof(Relais)), rList[i]);
  }
}

void setup() {
  Serial.begin(115200); 
  EEPROM.begin(1024); 
  loadFromEEPROM();
  
  for(int i=0; i<4; i++) { 
    pinMode(relayPins[i], OUTPUT); 
    digitalWrite(relayPins[i], LOW); 
  }
  
  //BME280 Adresse
  Wire.begin(I2C_SDA, I2C_SCL); 
  bme.begin(0x77); 
  ds18.begin();
  
  WiFi.softAP("ESP32 BBQ", "kaltrauch");
  if(strlen(config.ssid) > 1) WiFi.begin(config.ssid, config.pass);
  
  server.on("/", []() { server.send_P(200, "text/html", INDEX_HTML); });
  
  server.on("/data", []() {
    StaticJsonDocument<1500> doc;
    doc["t"] = getTemp(); doc["active"] = systemAktiv;
    doc["run"] = formatTime(millis() - startZeit);
    doc["tLeft"] = (timerEndeMillis > millis()) ? formatTime(timerEndeMillis - millis()) : "";
    JsonObject c = doc.createNestedObject("conf");
    c["sT"] = config.sensorType; c["rC"] = config.relayCount; c["ssid"] = config.ssid;
    JsonArray rels = doc.createNestedArray("rel");
    for(int i=0; i < config.relayCount; i++) {
      JsonObject r = rels.createNestedObject();
      r["n"] = rList[i].name; r["m"] = rList[i].mode; r["e"] = rList[i].ein; r["o"] = rList[i].aus; r["s"] = rList[i].aktuellAn;
    }
    String out; serializeJson(doc, out); server.send(200, "application/json", out);
  });
  
  server.on("/stop", []() { systemAktiv = false; allOff(); server.send(200); });
  server.on("/start", []() { systemAktiv = true; startZeit = millis(); timerEndeMillis = 0; server.send(200); });
  
  server.on("/setTimer", []() {
    timerDauerMinuten = server.arg("m").toInt();
    timerEndeMillis = (timerDauerMinuten > 0) ? millis() + (timerDauerMinuten * 60000) : 0;
    server.send(200);
  });
  
  server.on("/saveRel", []() {
    int num = server.arg("num").toInt();
    for(int i=0; i<num; i++) {
      strncpy(rList[i].name, server.arg("n"+String(i)).c_str(), 16);
      rList[i].mode = server.arg("m"+String(i)).toInt();
      rList[i].ein = server.arg("e"+String(i)).toFloat();
      rList[i].aus = server.arg("o"+String(i)).toFloat();
    }
    saveToEEPROM(); server.send(200);
  });
  
  server.on("/saveSys", []() {
    config.sensorType = server.arg("st").toInt();
    config.relayCount = server.arg("rc").toInt();
    strncpy(config.ssid, server.arg("s").c_str(), 32);
    strncpy(config.pass, server.arg("p").c_str(), 64);
    saveToEEPROM(); server.send(200);
    restartRequested = true; restartTimestamp = millis();
  });
  
  server.begin();
}

void loop() {
  if (restartRequested && (millis() - restartTimestamp > 2000)) ESP.restart();
  server.handleClient();
  
  if (timerEndeMillis > 0 && millis() > timerEndeMillis) { 
    systemAktiv = false; 
    timerEndeMillis = 0; 
    allOff(); 
  }
  
  static unsigned long lastReg = 0;
  if(systemAktiv && millis() - lastReg > 2000) {
    lastReg = millis();
    float t = getTemp();
    for(int i=0; i < config.relayCount; i++) {
      if(rList[i].mode == 0) { // Heizmodus
        if(t <= rList[i].ein) rList[i].aktuellAn = true; 
        else if(t >= rList[i].aus) rList[i].aktuellAn = false;
      } else { // Kühlmodus
        if(t >= rList[i].ein) rList[i].aktuellAn = true; 
        else if(t <= rList[i].aus) rList[i].aktuellAn = false;
      }
      digitalWrite(relayPins[i], rList[i].aktuellAn ? HIGH : LOW);
    }
  }
}

/* PINHEADER 4-Ralay WiFi Modul
 * 
 *         DS18              BMP280                   MAX6675
 *          x x        GND     o x                      x o  GPIO23
 *          x x        GPIO22  o x                      x x
 *          x x                x o  GPIO21              x x
 *     GND  o x                x x                 GND  o o  GPIO19
 *          x x                x x              GPIO18  o x
 *          x x                x x                      x x
 *  GPIO 4  o x                x x                      x x
 *          x x                x x                      x x
 *          x x                x x                      x x
 *          x o  3,3V          x o  3,3V                x o  3,3V
 */
